<?php

/*
 * This file is part of the SgDatatablesBundle package.
 *
 * (c) stwe <https://github.com/stwe/DatatablesBundle>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Sg\DatatablesBundle\Datatable\Column;

use Exception;
use Sg\DatatablesBundle\Datatable\Action\MultiselectAction;
use Sg\DatatablesBundle\Datatable\RenderIfTrait;
use Symfony\Component\OptionsResolver\OptionsResolver;

class MultiselectColumn extends ActionColumn
{
    // Render a Checkbox only if conditions are TRUE.
    use RenderIfTrait;

    /**
     * HTML <input> Tag attributes (except 'type' and 'value').
     * Default: null.
     *
     * @var array|null
     */
    protected ?array $attributes;

    /**
     * A checkbox value, generated by column name.
     * Default: 'id'.
     *
     * @var string
     */
    protected string $value;

    /**
     * Use the Datatable-Name as prefix for the value.
     * Default: false.
     *
     * @var bool
     */
    protected bool $valuePrefix;

    /**
     * Id selector where all multiselect actions are rendered.
     * Default: null ('sg-datatables-{{ sg_datatables_view.name }}-multiselect-actions').
     *
     * @var string|null
     */
    protected ?string $renderActionsToId;

    //-------------------------------------------------
    // ColumnInterface
    //-------------------------------------------------

    /**
     * {@inheritdoc}
     */
    public function isUnique(): bool
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function getOptionsTemplate(): string
    {
        return '@SgDatatables/column/multiselect.html.twig';
    }

    /**
     * {@inheritdoc}
     */
    public function addDataToOutputArray(array &$row): null|static
    {
        $row['sg_datatables_cbox'] = $this->callRenderIfClosure($row);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function renderSingleField(array &$row): static
    {
        $value = $row[$this->value];

        if (\is_bool($value)) {
            $value = (int) $value;
        }

        if (true === $this->valuePrefix) {
            $value = 'sg-datatables-'.$this->getDatatableName().'-checkbox-'.$value;
        }

        $row[$this->getIndex()] = $this->twig->render(
            $this->getCellContentTemplate(),
            [
                'attributes' => $this->attributes,
                'value' => $value,
                'start_html' => $this->startHtml,
                'end_html' => $this->endHtml,
                'render_if_cbox' => $row['sg_datatables_cbox'],
            ]
        );

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getCellContentTemplate(): string
    {
        return '@SgDatatables/render/multiselect.html.twig';
    }

    /**
     * {@inheritdoc}
     */
    public function allowedPositions(): ?array
    {
        return [0, self::LAST_POSITION];
    }

    /**
     * {@inheritdoc}
     */
    public function getColumnType(): string
    {
        return parent::MULTISELECT_COLUMN;
    }

    //-------------------------------------------------
    // Options
    //-------------------------------------------------

    /**
     * Configure options.
     *
     * @return $this
     */
    public function configureOptions(OptionsResolver $resolver): static
    {
        parent::configureOptions($resolver);

        // predefined in the view as Checkbox
        $resolver->remove('title');

        $resolver->setDefaults([
            'attributes' => null,
            'value' => 'id',
            'value_prefix' => false,
            'render_actions_to_id' => null,
            'render_if' => null,
        ]);

        $resolver->setAllowedTypes('attributes', ['null', 'array']);
        $resolver->setAllowedTypes('value', 'string');
        $resolver->setAllowedTypes('value_prefix', 'bool');
        $resolver->setAllowedTypes('render_actions_to_id', ['null', 'string']);
        $resolver->setAllowedTypes('render_if', ['null', 'Closure']);

        return $this;
    }

    //-------------------------------------------------
    // Getters && Setters
    //-------------------------------------------------

    /**
     * @throws Exception
     *
     * @return $this
     */
    public function setActions(array $actions): static
    {
        if (\count($actions) > 0) {
            foreach ($actions as $action) {
                $this->addAction($action);
            }
        } else {
            throw new Exception('MultiselectColumn::setActions(): The actions array should contain at least one element.');
        }

        return $this;
    }

    /**
     * Add action.
     *
     * @return $this
     */
    public function addAction(array $action): static
    {
        $newAction = new MultiselectAction($this->datatableName);
        $this->actions[] = $newAction->set($action);

        return $this;
    }

    /**
     * @return array|null
     */
    public function getAttributes(): ?array
    {
        return $this->attributes;
    }

    /**
     * @param array|null $attributes
     *
     * @return $this
     *@throws Exception
     *
     */
    public function setAttributes(?array $attributes): static
    {
        $value = 'sg-datatables-'.$this->datatableName.'-multiselect-checkbox';

        if (\is_array($attributes)) {
            if (\array_key_exists('type', $attributes)) {
                throw new Exception('MultiselectColumn::setAttributes(): The type attribute is not supported.');
            }

            if (\array_key_exists('value', $attributes)) {
                throw new Exception('MultiselectColumn::setAttributes(): The value attribute is not supported.');
            }

            if (\array_key_exists('name', $attributes)) {
                $attributes['name'] = $attributes['name'].'[]';
            } else {
                $attributes['name'] = $value.'[]';
            }
            if (\array_key_exists('class', $attributes)) {
                $attributes['class'] = $value.' '.$attributes['class'];
            } else {
                $attributes['class'] = $value;
            }
        } else {
            $attributes['name'] = $value.'[]';
            $attributes['class'] = $value;
        }

        $this->attributes = $attributes;

        return $this;
    }

    /**
     * @return string
     */
    public function getValue(): string
    {
        return $this->value;
    }

    /**
     * @param string $value
     *
     * @return $this
     */
    public function setValue(string $value): static
    {
        $this->value = $value;

        return $this;
    }

    /**
     * Get value prefix.
     *
     * @return bool
     */
    public function isValuePrefix(): bool
    {
        return $this->valuePrefix;
    }

    /**
     * Set value prefix.
     *
     * @param bool $valuePrefix
     *
     * @return $this
     */
    public function setValuePrefix($valuePrefix): static
    {
        $this->valuePrefix = $valuePrefix;

        return $this;
    }

    /**
     * @return string|null
     */
    public function getRenderActionsToId(): ?string
    {
        return $this->renderActionsToId;
    }

    /**
     * @param string|null $renderActionsToId
     *
     * @return $this
     */
    public function setRenderActionsToId(?string $renderActionsToId): static
    {
        $this->renderActionsToId = $renderActionsToId;

        return $this;
    }
}
