#!/bin/bash

RETCODE=0

BASEDEFS="base_defs"
FUNCS="deploy_funcs.sh"
MYSQOPT=${MYSQOPT:-}
INITKEYS=${INITKEYS:-}
export MYSQOPT

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

. "$DIR/$BASEDEFS" || { RETCODE=2; ERRMSG="eURL update: Fatal: Definition file ($DIR/$BASEDEFS) not found."; exit 1; }
. "$DIR/$FUNCS" || { RETCODE=2; ERRMSG="eURL update: Fatal: Deploy functions ($DIR/$FUNCS) not found."; exit 1; }

read_defs "$ORIG_DEPLOY_STR" || { RETCODE=2; ERRMSG="eURL update: Fatal: Cannot run read_defs()."; exit 1; }

if [[ -z "$MYSQOPT" && -n "$DBPASS" ]] ; then
	MYSQOPT="-p$DBPASS"
fi

#MYTMPDIR="/tmp/${SHORTNAME}_ipax.$$/"
ACTDIR="tmp"
MYOLDDIR=$(pwd)
ERRMSG=""
MESSAGE=""
OPT_CERTS_REGEN="--regen-certs"
OPT_AGENT_REPACK="--repack-agents"
g_app_active=""

tidy_up() {
	cd "$MYOLDDIR" || return 1
	#rm -rf "$MYTMPDIR" 2>/dev/null
}

trap tidy_up EXIT

print_help() {
	echo -e "\n$NAME script to update agent's packages server URL/IP address.\n"
	echo -e "\tsyntax1: $0 <new_url> ... repackage agent archives, set server to <new_url>"
	echo -e "\tsyntax2: $0 $OPT_AGENT_REPACK ... force agent archives repackage, no change to server URL"
	echo -e "\tsyntax3: $0 $OPT_CERTS_REGEN ... force regeneration of server certificate (warning, can cause breaking device -> server communication)"
}

stop_if() {
	MYOPWD=$(pwd)
	[[ -n "$g_app_active" && 0 -eq "$g_app_active" ]] && echo "Stopping $SHORTNAME ..." && {
		$C_SYSTEMCTL stop "${DWARFG_NAM}.service"
		$C_SYSTEMCTL stop "${DWARFG_NAM}_${SNMP_GW}.service"
	}
}

start_if() {
	[[ -n "$g_app_active" && 0 -eq "$g_app_active" ]] && echo "Starting $SHORTNAME ..." && $C_SYSTEMCTL start "${DWARFG_NAM}.service" && $C_SYSTEMCTL start "${DWARFG_NAM}_${SNMP_GW}.service"
	cd "$MYOPWD" || return 1
}

regen_certs() {
	if [ -n "$1" ] || [[ -n "$INITKEYS" && "$USE_SSL" -gt 0 && "$UPGRADING" -eq 0 ]] ; then
		if [ -n "$NEW_URL" ] ; then
			CERTDOMAIN="$NEW_URL" "$BINDIR/gencerts.sh" || { RETCODE=2; ERRMSG="eURL update: Error during generating certificate"; }
		else
			"$BINDIR/gencerts.sh" || { RETCODE=2; ERRMSG="eURL update: Error during generating certificate"; }
		fi
		MESSAGE="${MESSAGE}Certificated were re-generated. "
		return 0
	elif [ "$UPGRADING" -ne 0 ] ; then
		return 0
	fi
	ERRMSG="Failed to match condition for certificate generation. Use $OPT_CERTS_REGEN to force it."
	return 1
}

update_url() {
	[ "$EXTERNURL" != "$NEW_URL" ] && {
		echo "EXTERNURL/1 $EXTERNURL/$NEW_URL"
		echo "Updating DEF_SERVER to $NEW_URL ..."
		update_externurl "$NEW_URL" || { RETCODE=1; ERRMSG="eURL update: Unable to update EXTERNURL in $BASEDEFS"; return 1; }
		$C_MYSQL $MYSQOPT "$DWARFG_DBN"<<EOF
UPDATE product_conf SET strval="$NEW_URL" WHERE name="ExternalURL";
EOF
	}
}

repack_agents() {
	g_serv_mindate="$(date "+%F")"
	while true; do
		if [[ -z "$1" && "$EXTERNURL" = "$NEW_URL" && "$UPGRADING" -eq 0 ]] ; then
			ERRMSG="Failed to match condition for agents repack. Use $OPT_AGENT_REPACK to force it."
			return 1
		else
			#mkdir $MYTMPDIR || { RETCODE=1; ERRMSG="eURL update: Unable to create tmp directory $MYTMPDIR"; break; }
			#cd $MYTMPDIR || { RETCODE=1; ERRMSG="eURL update: Unable to chdir to $MYTMPDIR"; break; }
			cd "$BINDIR" || { RETCODE=3; ERRMSG="directory access error during agent repack"; break; }
			for i in "$BINDIR/agents"/*/*tgz; do
				echo "Updating package $i:"
				mkdir $ACTDIR || { RETCODE=3; ERRMSG="eURL update: Agent repack: Unable to create $ACTDIR"; break 2; }
				cd $ACTDIR || { RETCODE=3; ERRMSG="eURL update: Agent repack: Unable to chdir to $ACTDIR"; break 2; }
				tar xzf "$i";
				find . -name "${SHORTNAME}_agent.sh" -type f -print0 | while IFS= read -r -d '' file; do
					FULLFILE="$(pwd)/$file"
					echo -e "\tupdating file $FULLFILE..."
					sed -i --follow-symlinks "0,/g_ctxt_defserv=.*/s/g_ctxt_defserv=.*/g_ctxt_defserv=$NEW_URL/" "$file"
					sed -i --follow-symlinks "0,/g_servid=.*/s/g_servid=.*/g_servid=\"$SERVID\"/" "$file"
					sed -i --follow-symlinks "0,/g_fix_mindate=.*/s/g_fix_mindate=.*/g_fix_mindate=\"$g_serv_mindate\"/" "$file"
					sed -i --follow-symlinks "0,/g_forced_security=.*/s/g_forced_security=.*/g_forced_security=\"$AGENT_FORCED_SECURITY\"/" "$file"
				done
				if [ -d "./$(basename "${i//.tgz}")" ] ; then
					cp "$BINDIR/ssl/server.pem" "./$(basename "${i//.tgz}")/" 2>/dev/null
				elif [ -d "./a${SHORTNAME}" ] ; then
					cp "$BINDIR/ssl/server.pem" "./a${SHORTNAME}/" 2>/dev/null
				else
					RETCODE=5
					ERRMSG="eURL update: Agent \"$i\" repack problem: unable to locate dir to put server certificate to (no ./$(basename "${i//.tgz}") neither ./a${SHORTNAME} dirs there)"
					break
				fi
				tar --numeric-owner --group=0 --owner=0 -czf "$i" ./*;
				cd .. && rm -r "$ACTDIR"
			done
			for i in "$BINDIR/agents"/*/*ipk; do
				echo "Updating package $i:"
				mkdir $ACTDIR || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to create $ACTDIR"; break 2; }
				cd $ACTDIR || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to cd $ACTDIR"; break 2; }
				{ tar xzf "$i" && mkdir data control && pushd data >/dev/null && tar xzf ../data.tar.gz && { [ ! -f "$BINDIR/ssl/server.pem" ] || cp "$BINDIR/ssl/server.pem" "opt/a${SHORTNAME}/"; } && rm ../data.tar.gz && tar --numeric-owner --group=0 --owner=0 -czf ../data.tar.gz ./*; } || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to update server certificate in data package"; break; }
				{ popd >/dev/null && pushd control >/dev/null && tar xzf ../control.tar.gz && FULLFILE="$(pwd)/postinst" && echo -e "\tupdating file $FULLFILE..." ; } || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to unpack control part"; break; }
				sed -i --follow-symlinks "0,/g_ctxt_defserv=.*/s/g_ctxt_defserv=.*/g_ctxt_defserv=$NEW_URL/" "postinst"
				sed -i --follow-symlinks "0,/g_servid=.*/s/g_servid=.*/g_servid=\"$SERVID\"/" "postinst"
				{ rm ../control.tar.gz && tar --numeric-owner --group=0 --owner=0 -czf ../control.tar.gz ./* ; } || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to create control part."; break; }
				{ popd >/dev/null && tar --numeric-owner --group=0 --owner=0 -czf "$i" ./control.tar.gz ./data.tar.gz ./debian-binary; } || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to finish package."; break; }
				{ cd .. && rm -r "$ACTDIR"; } || { RETCODE=3; ERRMSG="eURL update: IPK repack ($i): Unable to cleanup"; break; }
			done
			sudo "$C_SYSTEMCTL" reload apache2.service || { RETCODE=4; ERRMSG="eURL update: URL updated but failed to reload apache configuration."; break; }
			MESSAGE="Server IP address was updated in agent archive to $NEW_URL."
			break
		fi
	done
}

validate_deploy_commands || { RETCODE=1; ERRMSG="eURL update: Unable to validate commands"; exit ; }

"$C_SYSTEMCTL" is-active "${DWARFG_NAM}.service" >/dev/null
g_app_active=$?

[ $# -ne 1 ] && print_help && exit 1
case $1 in
	'-h'|'--help')
		print_help
		exit 0
		;;
	"$OPT_AGENT_REPACK")
		NEW_URL="$EXTERNURL"
		stop_if
		repack_agents "yes"
		RETCODE=$?
		start_if
		;;
	"$OPT_CERTS_REGEN")
		NEW_URL="$EXTERNURL"
		stop_if
		regen_certs "use_the_force_luke"
		RETCODE=$?
		start_if
		;;
	*)
		NEW_URL="$1"
		stop_if
		if [ "$NEW_URL" = "$EXTERNURL" ] || update_url ; then
			if [ "$USE_SSL" -eq 0 ] || regen_certs ; then
				repack_agents
				RETCODE=$?
			else
				RETCODE=$?
				ERRMSG="${ERRMSG}Unable to generate certs but URL updated already.";
			fi
		else
			RETCODE=$?
			ERRMSG="${ERRMSG}Unable to update URL."
		fi
		start_if
		;;
esac


case $RETCODE in
	0)
		[ -n "$MESSAGE" ] && echo "$MESSAGE"
		event_log 304 MSG="$MESSAGE"
		exit 0
		;;
	1|3|4)
		event_log 301 MSG="$ERRMSG"
		echo "$ERRMSG" >&2
		exit 1
		;;
	2)
		event_log 302 MSG="$ERRMSG"
		echo "$ERRMSG" >&2
		exit 1
		;;
	*)
		event_log 302 MSG="Unknown error: $ERRMSG"
		echo "$ERRMSG" >&2
		exit 1
		;;
esac

