<?php

declare(strict_types=1);

namespace App\Datatables;

use App\Bundles\Sg\DatatablesBundle\Datatable\Editable\CustomTextEditable;
use App\Bundles\Sg\DatatablesBundle\Datatable\Filter\CustomSelectFilter;
use App\Datatables\Column\CustomArrayColumn;
use App\Datatables\Column\CustomDeviceFreshnessColumn;
use App\Datatables\Column\CustomDeviceStatusColumn;
use App\Datatables\Column\CustomIpAddressesArrayColumn;
use App\Datatables\Column\CustomMonitoringGroupArrayColumn;
use App\Entity\Device;
use App\Entity\Device\DeviceCustomField;
use App\Entity\VDevice;
use Sg\DatatablesBundle\Datatable\Column\ActionColumn;
use Sg\DatatablesBundle\Datatable\Column\Column;
use Sg\DatatablesBundle\Datatable\Column\LinkColumn;
use Sg\DatatablesBundle\Datatable\Column\MultiselectColumn;
use Sg\DatatablesBundle\Datatable\Filter\SelectFilter;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;

/**
 * Class DeviceDatatable
 * @package App\Datatables
 */
class DeviceDatatable extends BaseDatatable
{
    public const tableColumnName_SnmpName = 'snmpname';

    public const username_SnmpName = 'SNMP Name';

    public const showColumns = ['devId', 'name', 'devTypeName', 'OsVerOsTypeName', 'OsVerVersion', 'Labels',
                        'MonitoringGroups', 'StatsSyncStatusDescription', 'StatLiveMissCountDescription'];

    public const availableColumns = ['devId' => 'DevId', 'name' => 'Name', 'devTypeName' => 'Type',
        'OsVerOsTypeName' => "OS", 'OsVerVersion' => 'Version', 'Labels' => 'Labels', 'SystemLabels' => 'System Labels',
        'IpAddresses' => 'Ip Addresses', 'MonitoringGroups' => 'MonitoringGroups',
        'ConfigProfileAutomaticDevId' => 'Cloned From', 'StatsSyncStatusDescription' => 'Sync Status',
        'StatLiveMissCountDescription' => 'Freshness', 'SerialNumber' => 'Serial Number',
        'MacAddress' => 'Mac Address', self::tableColumnName_SnmpName => self::username_SnmpName,
        'HostName' => 'Host Name', 'Uuid' => 'UuId',
        'Iccid' => 'IccId', 'ActiveSim' => 'Active Sim', 'SignalStrength' => 'Signal Strength',
        'Temperature' => 'Temperature', 'AgentProfile' => 'Agent Profile', 'AuthProfile' => 'Auth Profile',
        'Note1' => 'Note 1', 'Note2' => 'Note 2', 'Note3' => 'Note 3',
        '_Number1' => 'Number 1', '_Number2' => 'Number 2', '_Number3' => 'Number 3', '_Number4' => 'Number 4', '_Number5' => 'Number 5',
        '_Number6' => 'Number 6', '_Number7' => 'Number 7', '_Number8' => 'Number 8', '_Number9' => 'Number 9',
        '_Float1' => 'Float 1', '_Float2' => 'Float 2', '_Float3' => 'Float 3', '_Float4' => 'Float 4', '_Float5' => 'Float 5',
        '_Float6' => 'Float 6', '_Float7' => 'Float 7', '_Float8' => 'Float 8', '_Float9' => 'Float 9',
        '_Text1' => 'Text 1', '_Text2' => 'Text 2', '_Text3' => 'Text 3', '_Text4' => 'Text 4', '_Text5' => 'Text 5',
        '_Text6' => 'Text 6', '_Text7' => 'Text 7', '_Text8' => 'Text 8', '_Text9' => 'Text 9'
    ];

    protected string $entity = VDevice::class;
    protected string $name = 'device_datatable';

    protected array $devices;

    protected array $syncStatuses;

    protected array $freshnessStatuses;

    protected array $deviceTypes;

    protected array $osTypes;

    protected array $osVersions;

    protected array $deviceLabels;

    protected array $deviceSystemLabels;

    protected array $deviceMonitoringGroups;

    protected array $agentProfiles;

    protected array $authProfiles;

    protected array $columns;
    protected array $deviceSimCards;

    /**
     * {@inheritdoc}
     */
    public function buildDatatable(array $options = array(), bool $parentCall = false): void
    {

        $this->initData($options);

        parent::buildDatatable($options);

        if (!$parentCall) {
            $this->extensions->set(
                [
                    'buttons' => [
                        'create_buttons' => [
                            $this->getRefreshButton(),
                        ],
                    ],
                ]
            );

            $actions[] = $this->getDeviceToMonitoringGroupAction();
            $actions[] = $this->getAssignLabelsAction();
            $actions[] = $this->getAssignFwVersion();
            $actions[] = $this->getAssignConfigProfile();
            $actions[] = $this->getAssignAgentProfile();
            $actions[] = $this->getRemoveConfigProfile();

            $this->getMultiselectActions($actions);
            $this->getColumns($this->columns);
            $this->columnBuilder

                // ->add('id', new ChildRow(), array('ajax' =>array('route' =>'ci_supply_by_chem', 'args' => array('id' => 'id')) ))
                /* ->add('id', Column::class, array(
                     'title' => 'Id',
                     'width' => '100%;',
                     'filter' => array(TextFilter::class,
                         array(
                             'cancel_button' => true,
                         ),
                     ),
                 ))*/

                ->add(null, ActionColumn::class, array(
                    'title' => $this->translator->trans('sg.datatables.actions.column_name'),
                    'width' => '100%;',
                    'class_name' => 'row_device_actions',
                    'actions' => array(
                        array(
                            'route' => 'device_detail',
                            'route_parameters' => array(
                                'deviceId' => 'id',
                            ),
                            'label' => '' . $this->translator->trans('sg.datatables.actions.detail'),
                            'icon' => self::ICON_DETAIL,
                            'attributes' => array(
                                'rel' => 'tooltip',
                                'title' => $this->translator->trans('sg.datatables.actions.detail'),
                                'class' => 'btn btn-primary btn-xs',
                                'role' => 'button',
                            ),
                        ),

                    ),
                ))

            ;
        }
    }

    public function initData(array $options): void
    {

        $this->columns = $options['columns'];
        $this->devices = $options['devices'];
        $this->syncStatuses = $options['syncStatuses'];
        $this->freshnessStatuses = $options['freshnessStatuses'];
        $this->deviceTypes = $options['deviceTypes'];
        $this->osTypes = $options['osTypes'];
        $this->osVersions = $options['osVersions'];
        $this->deviceLabels = $options['deviceLabels'];
        $this->deviceSystemLabels = $options['deviceSystemLabels'];
        $this->deviceSimCards = $options['deviceSimCards'];
        $this->deviceMonitoringGroups = $options['deviceMonitoringGroups'];
        $this->agentProfiles = $options['agentProfiles'];
        $this->authProfiles = $options['authProfiles'];
    }

    public function getMultiselectActions(array $actions, array $options = []): void
    {

        $this->columnBuilder->add(null, MultiselectColumn::class, [
                'start_html' => '<div class="start_checkboxes">',
                'end_html' => '</div>',
                'width' => '10px',
                'add_if' => function () {
                    return $this->authorizationChecker->isGranted('ROLE_ADMIN');
                },
                'value' => 'id',
                'value_prefix' => true,
                //'render_actions_to_id' => 'sidebar-multiselect-actions',
                'actions' => $actions,
            ]);
    }

    private function getDeviceToMonitoringGroupAction(): array
    {

        return [
            'icon' => 'fa fa-layer-group',
            'label' => 'Add to Monitoring group',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Add to Monitoring Group',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-devices-to-monitoring-group', //Element ID of modal dialog
            ],
            'start_html' => '<div class="input-group">',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getAssignLabelsAction(): array
    {

        return [
            'icon' => 'fa fa-tags',
            'label' => 'Assign Labels',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign labels',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-labels',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getAssignFwVersion(): array
    {

        return [
            'icon' => 'fa fa-file',
            'label' => 'Assign Fw Version',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign Fw',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-fw',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getAssignConfigProfile(): array
    {

        return [
            'icon' => 'fa fa-cog',
            'label' => 'Assign Config Profile',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign Config Profile',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-config',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getAssignAgentProfile(): array
    {

        return [
            'icon' => 'fa fa-file-lines',
            'label' => 'Assign Agent Profile',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Assign Agent Profile',
                'class' => 'btn btn-secondary btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#add-device-agent-profile',
            ],
            'start_html' => '',
            'end_html' => '',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    protected function getRemoveConfigProfile(): array
    {

        return [
            'route' => 'ajax_device_remove_config_profile',
            //'route_parameters' => [],
            'icon' => 'fa fa-times',
            'label' => 'Remove Config Profile',
            'attributes' => [
                'rel' => 'tooltip',
                'title' => 'Remove Config Profile',
                'class' => 'btn btn-danger btn-xs mb-1 mr-sm-2',
                'role' => 'button',
                'id' => '#remove-device-config',
            ],
            'start_html' => '',
            'end_html' => '</div>',
            'render_if' => function () {
                return $this->authorizationChecker->isGranted('ROLE_ADMIN');
            },
        ];
    }

    public function getColumns($showColumnNames): void
    {

        foreach ($showColumnNames as $columnName) {

            if(str_starts_with($columnName, '_Number')) {

                $number = (int)str_replace('_Number', '', $columnName);
                $this->getColumnNumber($number);

            }elseif(str_starts_with($columnName, '_Float')){

                $number = (int)str_replace('_Float', '', $columnName);
                $this->getColumnFloat($number);

            }elseif(str_starts_with($columnName, '_Text')){

                $number = (int)str_replace('_Text', '', $columnName);
                $this->getColumnText($number);

            }else{

                $this->getColumnByName($columnName);

            }

        }

       /* $this->getColumnNote1();
        $this->getColumnNote2();
        $this->getColumnNote3();*/

        /*$this->getColumnByName('devId');
        $this->getColumnByName('name');
        $this->getColumnByName('devTypeName');
        $this->getColumnByName('OsVerOsTypeName');
        $this->getColumnByName('OsVerVersion');
        $this->getColumnByName('Labels');
        $this->getColumnMonitoringGroups();*/

        //$this->getColumnByName('MacAddr');
        //$this->getColumnByName('Serial');
        //$this->getColumnByName('HostName');
        //$this->getColumnByName('Uuid');

       /* $this->getColumnByName('ConfigProfileAutomaticDevId');
        $this->getColumnByName('StatsSyncStatusDescription');
        $this->getColumnByName('StatLiveMissCountDescription');*/
    }

    private function getColumnByName(string $name): void
    {

        if (method_exists($this, 'getColumn' . $name)) {
            $this->{'getColumn' . $name}();
        }
    }

    private function getColumnDevId(): void
    {

        $this->columnBuilder->add(
            'devId',
            Column::class,
            [
                    'title' => 'DevId',
                    'width' => '100%;',
                    'filter' => [
                        TextFilter::class,[
                            'search_type' => 'like',
                            'cancel_button' => true,
                        ],
                    ],
                ]
        );
    }

    private function getColumnName(): void
    {

        $this->columnBuilder->add(
            'name',
            Column::class,
            [
                'title' => 'Name',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
                'editable' => [
                    CustomTextEditable::class,[
                        'url' => 'app_datatables_edit',
                        'maxlength' => 32,
                        'placeholder' => 'Edit value',
                        'empty_text' => 'None',
                        'editable_if' => function ($row) {
                            if ($this->getUser()) {
                                //if ($row['createdBy']['id'] == $this->getUser()->getId() or true === $this->isAdmin()) {
                                return true;
                                //};
                            }

                            return true;
                        },
                        'mode' => 'inline',
                    ],
                ],
            ]
        );
    }

    private function getColumnDevTypeName(): void
    {

        $this->columnBuilder->add(
            'devType.name',
            Column::class,
            [
                'title' => 'Type',
                'width' => '100%;',
                'searchable' => true,
                'orderable' => true,
                'filter' => [SelectFilter::class, [
                    'classes' => '',
                    'search_type' => 'eq',
                    'multiple' => false,
                    'select_options' => ['' => 'Any'] +
                        $this->getOptionsArrayFromEntities($this->deviceTypes, 'name', 'name'),
                    'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnOsVerOsTypeName(): void
    {

        $this->columnBuilder->add(
            'osVer.osType.name',
            Column::class,
            [
                'title' => 'OS',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class, [
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            $this->getOptionsArrayFromEntities($this->osTypes, 'name', 'name'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnOsVerVersion(): void
    {

        $this->columnBuilder->add(
            'osVer.version',
            Column::class,
            [
                'title' => 'Version',
                //'dql' => "device.osVerDesired",
                //'dql' => "CONCAT(device.devId, ' ', device.name)",
                //'join_type' => 'leftJoin'
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class, [
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            //$this->getOptionsArrayFromEntities($this->osVersions, 'version', 'version'),
                           ($this->osVersions),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnLabels(): void
    {

        $this->columnBuilder->add(
            'label',
            CustomArrayColumn::class,
            [
                'name' => 'labels',
                'title' => 'Labels',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({l}.label) FROM App:LabelDeviceMap {p}
                                JOIN App:Label {l} with {l}.id = {p}.labelId
                                WHERE {p}.device = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            $this->getOptionsArrayFromEntities($this->deviceLabels, 'id', 'label'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnIpAddresses(){

        /*$this->columnBuilder->add(
            'ipAddresses', CustomIpAddressesArrayColumn::class, [
                'name' => 'ipAdresses',
                'title' => 'IP Addresses',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({net}.label, '_', {net}.ipAddr, '_', COALESCE({net}.ip6Addr, 'NULL'))
                            FROM App:DeviceNetDev {net}
                            WHERE {net}.device = device AND {net}.active = true AND {net}.isLinkable = true
                             AND {net}.ipAddr is not NULL)",
                'type_of_field' => 'string',
                'searchable' => true,
                'orderable' => false,
                'filter' => [
                    TextFilter::class,[
                        'classes' => '',
                        'search_type' => 'like',
                       'cancel_button' => true,
                    ],
                ],
            ]
        );*/

        $this->columnBuilder->add(
            'netDevs', CustomIpAddressesArrayColumn::class, [
                'name' => 'ipAdresses',
                'title' => 'IP Addresses',
                'width' => '100%;',
                'type_of_field' => 'string',
                'searchable' => true,
                'orderable' => false,
                'filter' => [
                    TextFilter::class,[
                        'classes' => '',
                        'search_type' => 'like',
                        'cancel_button' => true,
                    ],
                ],
            ]
        );


    }

    private function getColumnSystemLabels(): void
    {

        $this->columnBuilder->add(
            'systemLabel',
            CustomArrayColumn::class,
            [
                'name' => 'systemLabels',
                'title' => 'System Labels',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({sl}.label) FROM App:SystemLabelDeviceMap {pa}
                                JOIN App:SystemLabel {sl} with {sl}.id = {pa}.labelId
                                WHERE {pa}.device = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            $this->getOptionsArrayFromEntities($this->deviceSystemLabels, 'id', 'label'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnNote1(): void
    {

        $this->getColumnNotes(1);
    }

    private function getColumnNote2(): void
    {

        $this->getColumnNotes(2);
    }

    private function getColumnNote3(): void
    {

        $this->getColumnNotes(3);
    }

    private function getColumnNumber1(): void
    {

        $this->getColumnNumber(1);
    }

    private function getColumnNotes(int $noteNum): void
    {

        $this->columnBuilder->add(
            'noteDevice.note' . $noteNum,
            Column::class,
            [
                'title' => 'Note ' . $noteNum,
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
                'editable' => [
                    CustomTextEditable::class,[
                        'url' => 'app_datatables_edit',
                        'maxlength' => 64, //maxlength custom implementation
                        'placeholder' => 'Edit value',
                        'empty_text' => 'None',
                        'editable_if' => function ($row) {
                            if ($this->getUser()) {
                                //if ($row['createdBy']['id'] == $this->getUser()->getId() or true === $this->isAdmin()) {
                                return true;
                                //};
                            }

                            return true;
                        },
                        'mode' => 'inline',
                    ],
                ],
            ]
        );
    }

    private function getColumnNumber(int $numberNum): void
    {

        $this->columnBuilder->add(
            'customFieldDevice.number' . $numberNum,
            Column::class,
            [
                'title' => 'Number ' . $numberNum,
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnFloat(int $numberNum): void
    {

        $this->columnBuilder->add(
            'customFieldDevice.float' . $numberNum,
            Column::class,
            [
                'title' => 'Float ' . $numberNum,
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnText(int $numberNum): void
    {

        $this->columnBuilder->add(
            'customFieldDevice.text' . $numberNum,
            Column::class,
            [
                'title' => 'Text ' . $numberNum,
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnMonitoringGroups(): void
    {

        $this->columnBuilder->add(
            'monitoringGroups',
            CustomMonitoringGroupArrayColumn::class,
            [
                'router' => $this->router,
                'name' => 'monitoringGroups',
                'title' => 'Monitoring Groups',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT(CONCAT_WS('_', {mg}.id, {mg}.name)) FROM App:MonitoringGroupMapping {mgmapping}
                                JOIN App:MonitoringGroup {mg} with {mg}.id = {mgmapping}.monitoringGroup
                                WHERE {mgmapping}.device = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                           $this->getOptionsArrayFromEntities($this->deviceMonitoringGroups),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnMacAddress(): void
    {

        $this->columnBuilder->add(
            'macAddr',
            Column::class,
            [
                'title' => 'MAC',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnSerialNumber(): void
    {

        $this->columnBuilder->add(
            'serial',
            Column::class,
            [
                'title' => 'SN',
                'default_content' => 'None',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnHostName(): void
    {

        $this->columnBuilder->add(
            'hostname',
            Column::class,
            [
                'title' => 'Hostname',
                'default_content' => 'None',
                //'dql' => " CONCAT(COALESCE (device.hostname,''), COALESCE (device.hostnameDesired, ''))",
                //'dql' => "device.hostname",
                'type_of_field' => 'string',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnSnmpname(): void
    {

        $this->getTextColumnByName(self::tableColumnName_SnmpName, self::username_SnmpName);
    }

    private function getColumnUuid(): void
    {

        $this->columnBuilder->add(
            'uuid',
            Column::class,
            [
                'title' => 'UUID',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnIccid(): void
    {

        $this->columnBuilder->add(
            'iccid',
            CustomArrayColumn::class,
            [
                'name' => 'deviceSimCards',
                'title' => 'Iccid',
                'width' => '100%;',
                'dql' => "(SELECT GROUP_CONCAT({sc}.iccid) FROM App\Entity\Tool\SimCard {sc}
                                WHERE {sc}.deviceUser = vdevice OR {sc}.deviceLastSeen = vdevice )",
                'type_of_field' => 'integer',
                'searchable' => true,
                'orderable' => true,
                'filter' => [
                    CustomSelectFilter::class,[
                        'search_type' => 'nothing',
                        'multiple' => false,
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsArrayFromEntities($this->deviceSimCards, 'id', 'iccid'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnActiveSim(): void
    {

        $this->columnBuilder->add(
            'activeSim',
            Column::class,
            [
                'title' => 'Active Sim',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnSignalStrength(): void
    {

        $this->columnBuilder->add(
            'stats.signalStrength',
            Column::class,
            [
                'title' => 'Signal Strength',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnTemperature(): void
    {

        $this->columnBuilder->add(
            'stats.temperature',
            Column::class,
            [
                'title' => 'Temperature',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnConfigProfileAutomaticDevId(): void
    {

        $this->columnBuilder->add(
            'configProfile.automatic.devId',
            LinkColumn::class,
            [
                'title' => 'Cloned from',
                'default_content' => '',
                'width' => '100%;',
                'route' => 'device_detail',
                'route_params' => function ($row) {
                    return [
                        'deviceId' => $row['configProfile']['automatic']['id'],
                    ];
                },
                'filter' => [
                    SelectFilter::class,[
                        'search_type' => 'eq',
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsFromArray($this->devices, 'devid', 'devid'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnStatsSyncStatusDescription(): void
    {

        $this->columnBuilder->add(
            'stats.syncStatus.description',
            CustomDeviceStatusColumn::class,
            [
                'title' => 'Sync Status',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any'] +
                            $this->getOptionsArrayFromEntities($this->syncStatuses, 'description', 'description'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnStatLiveMissCountDescription(): void
    {

        $this->columnBuilder->add(
            'stats.liveMissCount.description',
            CustomDeviceFreshnessColumn::class,
            [
                'title' => 'Freshness',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsArrayFromEntities($this->freshnessStatuses, 'description', 'description'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnAgentProfile(): void
    {

        $this->columnBuilder->add(
            'agentProfile.profileName',
            Column::class,
            [
                'title' => 'Agent Profile',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsArrayFromEntities($this->agentProfiles, 'profileName', 'profileName'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    private function getColumnAuthProfile(): void
    {

        $this->columnBuilder->add(
            'authProfile.profileName',
            Column::class,
            [
                'title' => 'Auth Profile',
                'default_content' => '',
                'width' => '100%;',
                'filter' => [
                    SelectFilter::class,[
                        'classes' => '',
                        'search_type' => 'eq',
                        'multiple' => false,
                        'select_options' => ['' => 'Any']
                            + $this->getOptionsArrayFromEntities($this->authProfiles, 'profileName', 'profileName'),
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }

    /**
     * @throws \Exception
     */
    private function getTextColumnByName(string $name, string $title, string $defaultContentValue = 'None'): void
    {

        $this->columnBuilder->add(
            $name,
            Column::class,
            [
                'title' => $title,
                'default_content' => $defaultContentValue,
                'type_of_field' => 'string',
                'width' => '100%;',
                'filter' => [
                    TextFilter::class,[
                        'cancel_button' => true,
                    ],
                ],
            ]
        );
    }
}
