<?php

declare(strict_types=1);

namespace App\Service;

use App\DataTable\DeviceDatatable;
use App\Entity\Product\ProductConf;
use App\Entity\Types\TypesValueType;
use App\ValueObject\Configuration;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;

class ConfigurationService
{
    public const RESULT_STR_VAL = 'strval';
    public const RESULT_INT_VAL = 'intval';
    public const RESULT_BOOL_VAL = 'boolval';

    /**
     * @var EntityManagerInterface
     */
    private EntityManagerInterface $entityManager;

    /**
     * @var EntityRepository
     */
    private EntityRepository $configRepository;

    /**
     * @var EntityRepository
     */
    private EntityRepository $valueTypeRepository;

    /**
     * Configuration constructor.
     * @param EntityManagerInterface $entityManager
     */
    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
        $this->configRepository = $this->entityManager->getRepository(ProductConf::class);
        $this->valueTypeRepository = $this->entityManager->getRepository(TypesValueType::class);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getProductRootDir(): string
    {
        return $this->getConfigItem(Configuration::CONFIG_PRODUCT_ROOT_DIR, self::RESULT_STR_VAL);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getServerIpAddress(): string
    {
        return $this->getConfigItem(Configuration::CONFIG_SERVER_IP_ADDRESS, self::RESULT_STR_VAL);
    }

    /**
     * @return int
     * @throws \Exception
     */
    public function getServerLocalPort(): int
    {
        return $this->getConfigItem(Configuration::CONFIG_SERVER_LOCAL_PORT, self::RESULT_INT_VAL);
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function isFirstRun(): bool
    {
        return (bool)$this->getConfigItem(Configuration::CONFIG_IS_FIRST_RUN, self::RESULT_INT_VAL);
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public function isSaasMode(): bool
    {
        return (bool)$this->getConfigItem(Configuration::CONFIG_SAAS_MODE, self::RESULT_INT_VAL);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getSupportLink(): string
    {
        return $this->getConfigItem(Configuration::CONFIG_SUPPORT_LINK, self::RESULT_STR_VAL);
    }

    /**
     * @return bool
     */
    public function isEnabledEmailNotifications(): bool
    {
        try {
            return (bool)$this->getConfigItem(Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATIONS, self::RESULT_INT_VAL);
        } catch (\Exception $exception) {
            return false;
        }
    }

    public function isEnabledEmailNotificationToAddress(): bool
    {
        try {
            return (bool)$this->getConfigItem(Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATION_TO_ADDRESS, self::RESULT_INT_VAL);
        } catch (\Exception $exception) {
            return false;
        }
    }

    /**
     * @return string|null
     */
    public function getNotificationEmail(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_NOTIFICATION_EMAIL, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSenderEmail(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SENDER_EMAIL, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSmtpServer(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_SERVER, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return int
     */
    public function getSmtpPort(): int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_PORT, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return 25;
        }
    }

    /**
     * @return int
     */
    public function getSmtpSecurity(): int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_SECURITY, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return 0; // None
        }
    }

    /**
     * @return string|null
     */
    public function getSmtpLogin(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_LOGIN, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSmtpPassword(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SMTP_PASSWORD, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSlackToken(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SLACK_TOKEN, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSlackChannel(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SLACK_CHANNEL, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getSysctlCommandPath(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SYSCTLPATH_COMMAND, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getServiceName(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SERVICE_CORE_NAME, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @return string|null
     */
    public function getAggregatedLogsFileName(): ?string
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_AGGREGATED_LOGS_FILE_NAME, self::RESULT_STR_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getDeviceTableColumns(): array
    {
        try {
            $data = $this->getConfigItem(Configuration::CONFIG_DEVICE_TABLE_COLUMNS, self::RESULT_STR_VAL);
            if (!is_null($data)) {
                return explode(',', $data);
            } else {
                return DeviceDatatable::showColumns;
            }
        } catch (\Exception $e) {
            return DeviceDatatable::showColumns;
        }
    }

    public function getSnmpGatewayThreads(): ?int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SNMP_GATEWAY_THREADS, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getSnmpGatewayBatch(): ?int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SNMP_GATEWAY_BATCH, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getSnmpGatewayTimeout(): ?int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SNMP_GATEWAY_TIMEOUT, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getSnmpGatewayRetries(): ?int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SNMP_GATEWAY_RETRIES, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getSnmpGatewayInterval(): ?int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SNMP_GATEWAY_INTERVAL, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getSnmpLogLevel(): ?int
    {
        try {
            return $this->getConfigItem(Configuration::CONFIG_SNMP_LOG_LEVEL, self::RESULT_INT_VAL);
        } catch (\Exception $e) {
            return null;
        }
    }

	public function getSnmpSslOverride(): ?int
	{
		try {
			return $this->getConfigItem(Configuration::CONFIG_SNMP_SSL_OVERRIDE_MODE, self::RESULT_INT_VAL);
		} catch (\Exception $e) {
			return null;
		}
	}

    public function setDeviceTableColumns(array $columns): void
    {

        $data = implode(',', $columns);
        $this->setConfigItem($data, Configuration::CONFIG_DEVICE_TABLE_COLUMNS, self::RESULT_STR_VAL);
    }

    /**
     * @param bool $isFirstRun
     * @return void
     */
    public function updateIsFirstRun(bool $isFirstRun): void
    {
        $this->setConfigItem($isFirstRun, Configuration::CONFIG_IS_FIRST_RUN, self::RESULT_BOOL_VAL);
    }

    /**
     * @param bool $isEnabled
     * @return void
     */
    public function updateIsEnabledEmailNotifications(bool $isEnabled): void
    {
        $this->setConfigItem($isEnabled, Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATIONS, self::RESULT_BOOL_VAL);
    }

    /**
     * @param bool $isEnabled
     * @return void
     */
    public function updateIsEnabledEmailNotificationToAddress(bool $isEnabled): void
    {
        $this->setConfigItem($isEnabled, Configuration::CONFIG_ENABLED_EMAIL_NOTIFICATION_TO_ADDRESS, self::RESULT_BOOL_VAL);
    }

    /**
     * @param string|null $email
     * @return void
     */
    public function updateNotificationEmail(?string $email): void
    {
        $this->setConfigItem($email, Configuration::CONFIG_NOTIFICATION_EMAIL, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $email
     * @return void
     */
    public function updateSenderEmail(?string $email): void
    {
        $this->setConfigItem($email, Configuration::CONFIG_SENDER_EMAIL, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $server
     * @return void
     */
    public function updateSmtpServer(?string $server): void
    {
        $this->setConfigItem($server, Configuration::CONFIG_SMTP_SERVER, self::RESULT_STR_VAL);
    }

    /**
     * @param int $port
     * @return void
     */
    public function updateSmtpPort(int $port): void
    {
        $this->setConfigItem($port, Configuration::CONFIG_SMTP_PORT, self::RESULT_INT_VAL);
    }

    /**
     * @param int $type
     * @return void
     */
    public function updateSmtpSecurity(int $type): void
    {
        $this->setConfigItem($type, Configuration::CONFIG_SMTP_SECURITY, self::RESULT_INT_VAL);
    }

    /**
     * @param string|null $login
     * @return void
     */
    public function updateSmtpLogin(?string $login): void
    {
        $this->setConfigItem($login, Configuration::CONFIG_SMTP_LOGIN, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $password
     * @return void
     */
    public function updateSmtpPassword(?string $password): void
    {
        if ($password !== null) {
            $this->setConfigItem($password, Configuration::CONFIG_SMTP_PASSWORD, self::RESULT_STR_VAL);
        }
    }

    /**
     * @param string|null $slackToken
     * @return void
     */
    public function updateSlackToken(?string $slackToken): void
    {
        $this->setConfigItem($slackToken, Configuration::CONFIG_SLACK_TOKEN, self::RESULT_STR_VAL);
    }

    /**
     * @param string|null $slackChannel
     * @return void
     */
    public function updateSlackChannel(?string $slackChannel): void
    {
        $this->setConfigItem($slackChannel, Configuration::CONFIG_SLACK_CHANNEL, self::RESULT_STR_VAL);
    }

    public function updateSnmpGatewayThreads(?int $threads): void
    {
        $this->setConfigItem($threads, Configuration::CONFIG_SNMP_GATEWAY_THREADS, self::RESULT_INT_VAL);
    }

    public function updateSnmpGatewayBatch(?int $batch): void
    {
        $this->setConfigItem($batch, Configuration::CONFIG_SNMP_GATEWAY_BATCH, self::RESULT_INT_VAL);
    }

    public function updateSnmpGatewayTimeout(?int $timeout): void
    {
        $this->setConfigItem($timeout, Configuration::CONFIG_SNMP_GATEWAY_TIMEOUT, self::RESULT_INT_VAL);
    }

    public function updateSnmpGatewayRetries(?int $retries): void
    {
        $this->setConfigItem($retries, Configuration::CONFIG_SNMP_GATEWAY_RETRIES, self::RESULT_INT_VAL);
    }

    public function updateSnmpGatewayInterval(?int $interval): void
    {
        $this->setConfigItem($interval, Configuration::CONFIG_SNMP_GATEWAY_INTERVAL, self::RESULT_INT_VAL);
    }

    public function updateSnmpLogLevel(?int $logLevel): void
    {
        $this->setConfigItem($logLevel, Configuration::CONFIG_SNMP_LOG_LEVEL, self::RESULT_INT_VAL);
    }

	public function updateSnmpSslOverride(?int $sslOverrideMode): void
	{
		$this->setConfigItem($sslOverrideMode, Configuration::CONFIG_SNMP_SSL_OVERRIDE_MODE, self::RESULT_INT_VAL);
	}

    /**
     * @param string $itemName
     * @param string $resultType
     * @return mixed
     * @throws \Exception
     */
    private function getConfigItem(string $itemName, string $resultType): mixed
    {

        $result = $this->configRepository->findOneBy(['name' => $itemName]);

        if (!$result) {
            throw new \Exception('No ' . $itemName . ' configuration found.');
        }

        return $result->{'get' . $resultType}();
    }

    /**
     * @param $value
     * @param $itemName
     * @param $resultType
     */
    private function setConfigItem($value, $itemName, $resultType): void
    {

        $config = $this->configRepository->findOneBy(['name' => $itemName]);

        if (!$config) {
            $config = new ProductConf();
            $config->setName($itemName);
            $config->setValType($this->valueTypeRepository->findOneBy(['id' => 3]));
        }

        $config->{'set' . $resultType}($value);

        $this->entityManager->persist($config);
        $this->entityManager->flush();
    }
}
