<?php

declare(strict_types=1);

namespace App\DataTable;

use App\Entity\ConfigProfile\ConfigProfile;
use Sg\DatatablesBundle\Datatable\Column\ActionColumn;
use Sg\DatatablesBundle\Datatable\Editable\TextEditable;
use Sg\DatatablesBundle\Datatable\Filter\TextFilter;
use Sg\DatatablesBundle\Datatable\Column\Column;

/**
 * Class LabelDatatable
 * @package App\Datatables
 */
class ConfigProfileDatatable extends BaseDatatable
{

    protected string $entity = ConfigProfile::class;
    protected string $name = 'config_profiles_datatable';

    /**
     * @param array $options
     * @throws \Exception
     */
    public function buildDatatable(array $options = []): void
    {

        parent::buildDatatable($options);

        $this->extensions->set([
            'buttons' => [
                'create_buttons' => [  // custom buttons
                    [
                        'action' => [
                            'template' => '/datatable/configProfile/createConfigProfileFromDevice.js.twig',
                        ],
                        'class_name' => 'btn-add',
                        'name' => 'addLabel',
                        'text' => '<i class=\"fa fa-display\"></i> Create From Device',
                    ],
					[
						'action' => [
							'template' => '/datatable/configProfile/createConfigProfileFromProfile.js.twig',
						],
						'class_name' => 'btn-info',
						'name' => 'addLabel',
						'text' => '<i class=\"fa fa-list\"></i> Create From existing Profile',
					],
					[
						'action' => [
							'template' => '/datatable/configProfile/uploadNewConfigProfile.js.twig',
						],
						'class_name' => 'btn-primary custom-btn',
						'name' => 'addLabel',
						'text' => '<i class=\"fa fa-upload\"></i> Upload new Profile',
					],
                    $this->getRefreshButton()
                ],
            ]
        ]);

        $this->addIdColumn();

        $this->columnBuilder
            ->add('name', Column::class, [
                'title' => 'Label',
                'width' => '80%;',
                'filter' =>[TextFilter::class,
                    [
                        'cancel_button' => true,
                    ],
                ],
                'editable' => [TextEditable::class,
                    [
                        'url' => 'app_datatables_edit',
                        'placeholder' => 'Edit value',
                        'empty_text' => 'None',
                        'editable_if' => function ($row) {
                            if ($this->getUser()) {
                                if(true === $this->isAdmin()) {
                                    return true;
                                };
                            }

                            return false;
                        },
                        'mode' => 'inline',
                    ],
                ],
            ])

			->add('Devices_count', Column::class, [
				'title' => 'Devices count',
				'width' => '100%;',
				'class_name' => 'row_devices_label_count',
				'dql' => '(SELECT COUNT({d}) FROM App\Entity\Device\Device {d} WHERE {d}.configProfile = configprofile.id)',
				'searchable' => true,
				'orderable' => true,
			])

            ->add(null, ActionColumn::class, options: [
                'title' => $this->translator->trans('sg.datatables.actions.title'),
                'width' => '100%;',
                'actions' => [

					[
						'route' => 'admin_config_profile_manage',
						'route_parameters' => [
							'configProfileId' => 'id',
						],
						'label' => $this->translator->trans('sg.datatables.actions.manage'),
						'icon' => self::ICON_MANAGE,
						'attributes' => [
							'rel' => 'tooltip',
							'title' => $this->translator->trans('sg.datatables.actions.manage'),
							'class' => 'btn btn-primary btn-xs mb-1 mr-sm-2',
							'role' => 'button',
							'id' => 'manage',
						],
						'start_html' => '<div class="input-group">',
						'render_if' => function ($row) {
							return $this->authorizationChecker->isGranted('ROLE_ADMIN');
						},
					],

					[
						'route' => 'admin_config_profile_assign_device',
						'route_parameters' => [
							'configProfileId' => 'id',
						],
						'label' => $this->translator->trans('sg.datatables.actions.devices'),
						'icon' => self::ICON_DETAIL,
						'attributes' => [
							'rel' => 'tooltip',
							'title' => $this->translator->trans('sg.datatables.actions.devices'),
							'class' => 'btn btn-primary btn-xs mb-1 mr-sm-2',
							'role' => 'button',
							'id' => 'devices',
						],
						'start_html' => '<div class="input-group">',
						'render_if' => function ($row) {
							return $this->authorizationChecker->isGranted('ROLE_ADMIN');
						},
					],

					[
						'route' => 'ajax_config_profile_delete',
						'route_parameters' => [
							'configProfileId' => 'id',
						],
						'label' => $this->translator->trans('sg.datatables.actions.delete'),
						'icon' => self::ICON_DELETE,
						'attributes' => [
							'rel' => 'tooltip',
							'title' => $this->translator->trans('sg.datatables.actions.delete'),
							'class' => 'btn btn-danger btn-xs mb-1',
							'role' => 'button',
							'data-toggle' => 'modal',
							'id' => 'delete',
						],
						'render_if' => function ($row) {
							return $this->authorizationChecker->isGranted('ROLE_ADMIN');
						},
						'end_html' => '</div>',
					],

                ],
            ]);

    }

}