#!/bin/bash
# Dwarfg deployment script
# Copyright (c) Dwarf Technologies, Jan Otte
# NOTE: while the file (itself) is copyrighted, you are invited to use the stuff you learn here anywhere you need

FUNCS="deploy_funcs.sh"

command -v dirname >/dev/null 2>&1 || {
	echo "\"dirname\" command not available, bailing out." >&2
	exit 1
}

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

. "$DIR"/"$FUNCS" || {
	echo "Unable to read deploy functions, exiting..." >&2
	exit 1
}

BASEDEFS="base_defs"
DWARFG_DUMP="app_data_backup.sql"
INTERACTIVE=1
BACKUP_RESTORE=0
USE_SSL=1
GET_NONDEB=0
UPGRADING="${UPGRADING:-0}"
PORT_OFFSET="${PORT_OFFSET:-0}"
DWARFGD_PORT="${DWARFGD_PORT:-8484}"
LICENSE_FILE="${LICENSE_FILE:-}"
PKG_INST_SCRIPT="machine_install.sh"
LICMAN="licman"
NEW_DB=""
SKIP_START=0

export UPGRADING

print_help() {
	echo -e "\nHi there, this is $NAME deploy script.\nThe purpose of this script is to deploy $NAME on your server,\npossibly wiping out existing installation or retaining current data.\n\nAllowed parameters:\n"
	echo -e "\t$PARAM_DOMAIN <domain> ... IF USING, PUT THIS PARAMETER *FIRST*. Install $NAME for a particular domain. This affects apache2 config and DB name. You need to use this parameter or specify IP address by 'export CERTDOMAIN=\"1.2.3.4\"' before calling the install script."
	echo -e "\t$PARAM_NONDEB ... After regular installation finishes, attempt to install also extra SW components outside of the Debian archive. These components are usually installed by dwarfg_grafint (Grafana) and dwarfg_inst_pyext (Python PureSNNMP library to enable SNMP Gateway and SSHWifty web-based terminal to allow access to the SSH tunnel directly from the browser). Read respective dwarfg_ commands help to find out how to install this when your deployment server is offline without access to Internet. Please note that in the case of multi-deployment machine, only one deployment can have automatic integration with Grafana enabled."
	echo -e "\t$PARAM_LICFILE <license_file_ABSOLUTE_path> ... Use license_file_path license instead of the one bundled in the package. NOTE: license check tool requires some libraries that are e.g. installed during deployment (if you have not installed $NAME on this machine, the deployment may fail as the library is not present. Either install the libraries manually, or install/uninstall $NAME without using this parameter first)"
	echo -e "\t$PARAM_OFFSET <number> ... For usage together with domain - a number to offset tunnel ports with (usually a 100 with every additional domain). NOTE: this is now set automatically, no need to specify this explicitely."
	echo -e "\t$PARAM_SERVSSH <number> ... tell agents to contact server regarding tunnels on non-standard SSH port (other than 22)"
	echo -e "\t$PARAM_SECURITY [0..2] ... Specify global agent forced security level. 2 (default) means check server certificate always, do not allow exchange. 1 means check but allow exchange if security tokens matches. 0 means do not check server certificate (use case = devices and server on VPN, internal CA not trusted by endpoint devices). Not changeable after deployment."
	echo -e "\t$PARAM_DWARFGD_PORT <number> ... set local port $NAME daemon is using to listen. Propagates to Apache config. NOTE: this is set automatically, no need to specify explicitly."
	echo -e "\t$PARAM_ADMNAME <login_email> ... Set (the default) administrator login (e-mail) to <login_email>."
	echo -e "\t$PARAM_ADMPWD <password> ... Set (the default) administrator password to <password>."
	echo -e "\t$PARAM_OPERNAME <login_email> ... Set (default) operator login (e-mail) to <login_email>."
	echo -e "\t$PARAM_OPERPWD <password> ... Set (default) operator password to <password>."
	echo -e "\t$PARAM_SERVID <servid> ... Provide your own server ID string. Useful for testing purposes and disaster recovery only."
	echo -e "\t$PARAM_NOSSL ... turn off SSL (if you use a SSL-facing proxy like nginx or caddy)"
	echo -e "\t$PARAM_BR ... Activate backup+restore during deploy. Either both data+config or just config will be retained, based on new version prerequisities."
	echo -e "\t$PARAM_BONLY ... Prepare backup (into ./$DWARFG_DUMP) and exit immediately (do not cleanup, do not deploy)"
	echo -e "\t$PARAM_RONLY ... Attempt to restore data only, neither backup nor deploy. Backup is expected in ./$DWARFG_DUMP"
	echo -e "\t$PARAM_FORCE ... Non-interactive/force mode: wipes any existing installation if found (data and config lost)."
	echo -e "\t$PARAM_UPGRADE ... Perform in-place upgrade. Domain is taken from $PARAM_DOMAIN"
	echo
}

process_params() {
	while [ $# -gt 0 ] ; do
		case "$1" in
			"$PARAM_BONLY")
				echo "Attempting to backup (for reinstall of the SAME version) to ./$DWARFG_DUMP"
				cbackup
				exit $?
				;;
			"$PARAM_RONLY")
				echo "Attmpting to restore (after reinstall of the SAME version) from ./$DWARFG_DUMP..."
				crestore_data
				exit $?
				;;
			"$PARAM_BR")
				BACKUP_RESTORE=1
				cbackup
				;;
			"$PARAM_FORCE")
				INTERACTIVE=0
				:
				;;
			"$PARAM_UPGRADE")
				UPGRADING=1
				;;
			"$PARAM_NOSTART")
				SKIP_START=1
				;;
			"$PARAM_LICFILE")
				if [ -z "$2" ] ; then
					fail_fat "License parameter ($PARAM_LICFILE) was used without a path to the license."
				fi
				if [ ! -f "$2" ] ; then
					fail_fat "License parameter target ($2) does not seem to be a file."
				fi
				LICENSE_FILE="$2"
				if [ -x "$DIR/$LICMAN" ] ; then
					:
				else
					fail_fat "Custom license provided but unable to locate license manager binary ($LICMAN)."
				fi
				shift
				;;
			"$PARAM_DOMAIN")
				if [ -z "$2" ] ; then
					fail_fat "DOMAIN parameter ($PARAM_DOMAIN) has to be followed by domainname."
				fi
				update_domain "$2"
				shift
				;;
			"$PARAM_SECURITY")
				if [ -z "$2" ] || [[ "$2" != "0" && "$2" != "1" && "$2" != "2" ]] ; then
					fail_fat "Agent security parameter ($PARAM_SECURITY) has to be followed by number from 0 to 2."
				fi
				update_agent_security "$2"
				shift
				;;
			"$PARAM_OFFSET")
				if [ -z "$2" ] ; then
					fail_fat "Offset parameter ($PARAM_OFFSET) has to be followed by a number."
				fi
				PORT_OFFSET=$2
				update_offset "$2"
				shift
				;;
			"$PARAM_SERVSSH")
				if [ -z "$2" ] ; then
					fail_fat "Server ssh parameter ($PARAM_SERVSSH) has to be followed by a number."
				fi
				update_servssh "$2"
				shift
				;;
			"$PARAM_NOSSL")
				USE_SSL=0
				;;
			"$PARAM_NONDEB")
				GET_NONDEB=1
				;;
			"$PARAM_SERVID")
				if [ -z "$2" ] ; then
					fail_fat "Manual Servid parameter ($PARAM_SERVID) has to be followed by an identifier."
				fi
				update_servid "$2"
				shift
				;;
			"$PARAM_DWARFGD_PORT")
				if [ -z "$2" ] ; then
					fail_fat "$NAME daemon port ($PARAM_DWARFGD_PORT) has to be followed by a number."
				fi
				update_daemon_port "$2"
				echo "WARNING, $NAME daemon port was updated in the default $BASEDEFS, take care when reusing this deploy directory for reinstall (port is computed automatically, offsetting the value from $BASEDEFS)."
				shift
				;;
			"$PARAM_ADMNAME")
				if [ -z "$2" ] ; then
					fail_fat "Admin login (argument to the $PARAM_ADMNAME) is missing."
				fi
				ADMNAME="$2"
				export ADMNAME
				shift
				;;
			"$PARAM_ADMPWD")
				if [ -z "$2" ] ; then
					fail_fat "Admin password (argument to the $PARAM_ADMPWD) is missing."
				fi
				ADMPWD="$2"
				export ADMPWD
				shift
				;;
			"$PARAM_OPERNAME")
				if [ -z "$2" ] ; then
					fail_fat "Operator login (argument to the $PARAM_OPERNAME) is missing."
				fi
				OPERNAME="$2"
				export OPERNAME
				shift
				;;
			"$PARAM_OPERPWD")
				if [ -z "$2" ] ; then
					fail_fat "Operator password (argument to the $PARAM_OPERPWD) is missing."
				fi
				OPERPWD="$2"
				export OPERPWD
				shift
				;;
			*)
				echo "Unrecognized option: $1, printing help..."
				print_help
				exit 1
				;;
		esac
		shift
	done
}

prep_python_venv() {
	echo -n "Preparing python virtual environment at $PYTHON_VENV..."
	python3 -m venv --system-site-packages "$PYTHON_VENV" || { echo "FAILURE"; return 1; }
	echo " success";
}

read_defs
echo "$NAME application deploy script."
trap remove_deploy EXIT
if [[ -n "$DOMAIN" && 0 -eq "$UPGRADING" ]] ; then
	if [ 0 -ne $INTERACTIVE ] ; then
		echo
		echo "Warning, the DOMAIN parameter read from config is non-zero: $DOMAIN"
		echo "This means the existing install package was used to install $NAME for a particular domain."
		echo "You can clear the DOMAIN variable in $BASEDEFS manually or use --domain paramater to set."
		echo "Ignore this if you are using --domain parameter to define domain right now."
		echo
		echo "ENTER to proceed, CTRL-C to cancel deploy."
		read -r
	fi
fi
if [[ -z "$DOMAIN" && 1 -eq "$UPGRADING" ]] ; then
	echo
	echo "Warning, UPGRADING mode requested but DOMAIN is set to empty."
	echo "This indicates an error."
	echo "Now would be a good time to interrupt upgrade by CTRL-C and restore from backup."
	echo "If you insist on continuing, press ENTER to proceed."
	read -r
fi
if [ $# -ne 0 ] ; then
	process_params "$@"
fi
read_defs
[[ -z "$DOMAIN" && -z "$EXTERNURL" && -z "$CERTDOMAIN" ]] && {
	fail_fat "Both DOMAIN and EXTERNURL are empty. Use $PARAM_DOMAIN or export IP address as CERTDOMAIN variable."
}
IFS='.' read -a DEBIAN_VERSION < /etc/debian_version
MYID=$($C_ID -u)
[ 0 -ne "$MYID" ] && fail_fat "$NAME deploy script must run under root."
# first get options coming from Dwarfg deployment number
if [ 1 -eq "$UPGRADING" ]; then
	upgrade_deploy || fail_fat "The deployment to be upgraded is not valid/available"
else
	reserve_deploy || fail_fat "The selected $NAME deployment domain ($DOMAIN) is not available. Conflicting lock is $DEPLOY_BLOCK"
# compute all ports based on $TARGET_DEPLOY_NUM (set by reserve_deploy and upgrade_deploy)
fi
[ 0 -eq "$UPGRADING" ] && {
	[ 8484 -ne "$DWARFGD_PORT" ] && {
		echo -e "WARNING, the default DWARFGD_PORT does not equal 8484.\nDeploy script is about to recompute the DWARFGD_PORT and starting from non-default value may result in incorrect port, potentially clashing other deployed $NAME installations.\nHit CTRL-C to cancel deployment now."
		read
	}
	PORT_OFFSET=$((OFFSET_SHIFT*(TARGET_DEPLOY_NUM-1)))
	DWARFGD_PORT=$((DWARFGD_PORT+TARGET_DEPLOY_NUM-1))
	echo "Daemon port calculated to $DWARFGD_PORT"
}
if [ -f "$DWARFG_SITECONF" ] ; then
	echo "$NAME site configuration ($DWARFG_SITECONF) found, sourcing it..."
	. "$DWARFG_SITECONF"
fi
echo -n "Checking + installing prerequisities (pwd: "
pwd
echo ") ..."
if [ -f "$DIR/$PKG_INST_SCRIPT" ] ; then
	"$DIR/$PKG_INST_SCRIPT" run
	if [ $? -ne 0 ] ; then
		fail_fat "Unable to install required packages. Run $PKG_INST_SCRIPT manually to resolve, then rerun deploy. NOTE that this version of $NAME is fit to install on Debian GNUI/Linux 12 Bookworm. If you want to install it on different version/OS, you would need to customize package dependencies - also, running on other version/OS may potentially introduce problems and are not supported."
	fi
else
	fail_fat "Unable to find prerequisities installation script ($PKG_INST_SCRIPT)."
fi
validate_deploy_commands || fail_fat "Some required commands are missing"
if [ 1 -eq "$UPGRADING" ] ; then
	[ -f "$BINDIR"/upgrading ] || fail_fat "Supposed upgrade mode not confirmed by $BINDIR/upgrading"
else
	echo "Checking existing installation...($BINDIR, $SRVDIR)"
	[[ ( -e "$BINDIR" ) || ( -e "$SRVDIR" ) ]] && wipeout "$BINDIR or $SRVDIR exists" && echo "Removed remnants from previous installation. Exiting now, please re-run the installation." && exit 1
fi
"$C_ID" -u "$WWWUSER" >/dev/null 2>&1 || fail_fat "WWW user ($WWWUSER) not detected."
echo "Resolving users..."
add_app_group || fail_fat "Unable to add $NAME group ($APPGROUP)"
add_app_user || fail_fat "Unable to add user $APPUSER"
add_utg || fail_fat "Unable to add user $APPUSER to the group $WMGROUP"
add_app_home || fail_fat "Unable to prepare $NAME home directory ($DWARFG_HOME)"
if [ -z "$DBPASS" ] ; then
	new_dbpass || fail_fat "Unable to generate new DB password"
	NEW_DB="yes"
fi
"$C_MYSQL" $MYSQOPT <<EOF
exit
EOF
if [ $? -ne 0 ] ; then
	fail_fat "Unable to run mysql under root. You can specify your password in the MYSQL_PWD enviroment variable"
fi
echo "Copying application data..."
{ [ -n "$BINDIR" ] && "$C_MKDIR" -p "$BINDIR"; } || fail_fat "Could not create $BINDIR"
{ [ -n "$SRVDIR" ] && "$C_MKDIR" -p "$SRVDIR"; } || fail_fat "Could not create $SRVDIR"
"$C_CP" -dr "$DIR"/!(_*) "$BINDIR"/ || fail_fat "Could not copy application to $BINDIR"
update_install_paths "$BINDIR" || fail_fat "Could not update install paths..."
[ 0 -eq "$UPGRADING" ] && {
	update_offset "$PORT_OFFSET" "$ORIG_DEPLOY_STR"
	update_daemon_port "$DWARFGD_PORT" "$ORIG_DEPLOY_STR"
}
add_app_service || fail_fat "Unable to install $NAME service into the system (/etc/systemd/system/${DWARFG_NAM}.service)"
if [ -n "$LICENSE_FILE" ] ; then
	cp "$LICENSE_FILE" "$BINDIR/${SHORTNAME}_license.lic" || fail_fat "Failed to inject license $LICENSE_FILE"
fi
"$C_CHOWN" -R "$APPUSER" "$BINDIR" || fail_fat "Unable to set permissions for $BINDIR"
"$C_CHOWN" root "$BINDIR/${SHORTNAME}_logrotate.conf"
# prepare Dwarfg service checking and management for web user
add_app_sudoers || fail_fat "Unable to prepare $NAME sudo rules."
echo "*/2 * * * * $APPUSER $BINDIR/rssgen.sh >/dev/null 2>&1" >"/etc/cron.d/$DWARFG_DBN"
echo "11 0 * * * $APPUSER $BINDIR/count_traffic_daily.sh >/dev/null 2>&1" >"/etc/cron.d/$DWARFG_DBN"
echo "Preparing runtime data directories..."
"$C_MKDIR" -p "$SRVDIR_SRV" "$SRVDIR_CLI" "$LOGDIR" "$DATADUMPDIR" "$SRVDIR_FW" "$SRVDIR_CONFIG" "$FWUPLOADDIR" "$LICUPLOADDIR" "$SECRETSDIR" "$KEYSDIR" "$PWFILESDIR" "$SYSDEVSDIR" || fail_fat "Could not create some of following directories: $SRVDIR_SRV, $SRVDIR_CLI, $LOGDIR, $DATADUMPDIR, $SRVDIR_FW, $SRVDIR_CONFIG, $FWUPLOADDIR, $LICUPLOADDIR, $SECRETSDIR, $KEYSDIR, $PWFILESDIR, $SYSDEVSDIR"
"$C_CHMOD" -R go= "$SECRETSDIR" || fail_fat "Unable to set correct permissions for $SECRETSDIR"
"$C_CHOWN" -R "$APPUSER" "$SECRETSDIR" || fail_fat "Unable to change ownership of $SECRETSDIR"
"$C_CHOWN" -R "$WWWUSER" "$KEYSDIR" "$PWFILESDIR" "$LICUPLOADDIR" || fail_nof "Could not chown one of: $KEYSDIR, $PWFILESDIR, "$LICUPLOADDIR" push functionality from GUI will not work fully."
"$C_CHGRP" -R "$APPGROUP" "$LICUPLOADDIR" || fail_fat "Unable to change owner of $LICUPLOADDIR"
"$C_CHMOD" "=770" "$LICUPLOADDIR" || fail_fat "Unable to set correct permissions for $LICUPLOADDIR"
for i in $FW_FLAVORS; do
	"$C_MKDIR" -p "$SRVDIR_FW/$i" || fail_fat "Could not create directory $SRVDIR_FW/$i"
done
"$C_CHOWN" -R "$APPUSER" "$SRVDIR_SRV" || fail_fat "Unable to change ownership of $SRVDIR_SRV"
"$C_CHOWN" -R "$APPUSER" "$DWARFG_HOME/.ssh" || fail_fat "Unable to change ownership of $DWARFG_HOME/.ssh"
"$C_CHOWN" -R "$WWWUSER" "$SRVDIR_CLI" || fail_fat "Unable to change ownership of $SRVDIR_CLI"
"$C_CHOWN" -R "$APPUSER" "$SRVDIR_FW" || fail_fat "Unable to change ownership of $SRVDIR_FW"
"$C_CHOWN" -R "$APPUSER" "$SRVDIR_CONFIG" || fail_fat "Unable to change ownership of $SRVDIR_CONFIG"
"$C_TOUCH" "$LOGDIR/$DWARFG_LOG" || fail_fat "Unable to create logfile for $NAME ($LOGDIR/$DWARFG_LOG)"
echo 0 >"$LOGDIR/$PIDLOG" || fail_fat "Unable to create pidfile ($LOGDIR/$PIDLOG)"
"$C_CHOWN" "$APPUSER" "$LOGDIR/$DWARFG_LOG" || fail_fat "Unable to change ownership of $LOGDIR/$DWARFG_LOG"
"$C_CHOWN" "$APPUSER" "$DATADUMPDIR" || fail_fat "Unable to change ownership of $DATADUMPDIR"
"$C_CHOWN" "$APPUSER" "$LOGDIR/$PIDLOG" || fail_fat "Unable to change ownership of $LOGDIR/$PIDLOG"
"$C_CHMOD" "=775" "$LOGDIR" || fail_fat "Unable to set correct permissions on $LOGDIR"
"$C_CHGRP" "$APPGROUP" "$LOGDIR" || fail_fat "Unable to change group owner on $LOGDIR"
"$C_CHOWN" "$APPUSER" "$SRVDIR_FW" || fail_fat "Unable to change ownership for $SRVDIR_FW"
add_apache_links || fail_fat "Unable to correctly link $NAME web from Apache HTML tree."
echo "Running post-install app data init script..."
[ 1 -eq "$BACKUP_RESTORE" ] && crestore_preconfig
[ 0 -eq "$UPGRADING" ] && {
	update_usessl "$USE_SSL" "$ORIG_DEPLOY_STR"
}
export USE_SSL
PPV=$(declare -f prep_python_venv)
export PYTHON_VENV
sudo -E -u "$APPUSER" bash -c "$PPV; prep_python_venv" || fail_fat "Unable to prepare Python virtual environment $PYTHON_VENV"
"$DIR/$AFTERINST" "$BINDIR" || fail_fat "Unable to finish $NAME initialization"
# update service name so that GUI knows what to execute for service start/restart
"$C_MYSQL" $MYSQOPT "$DWARFG_DBN" <<EOF
UPDATE product_conf SET strval="${DWARFG_NAM}" WHERE name='ServiceCoreName';
UPDATE product_conf SET strval="${DWARFG_NAM}_${SNMP_GW}" WHERE name='ServiceSNMPGatewayName';
EOF
[ 1 -eq "$BACKUP_RESTORE" ] && crestore_postconfig
[ 1 -eq "$BACKUP_RESTORE" ] && crestore_data
create_app_fwdir || fail_fat "Unable to prepare $NAME firmware download directory."
DEPLOY_SUCCESS=1
if [ "$UPGRADING" -eq 1 ] ; then
	echo "Skipping $NAME enable+start due to upgrade detected. Upgrade script should enable + start $NAME."
	event_log 307 MSG="$NAME version $VERSION (upgrade mode) was successfully deployed."
	exit 0
else
	if [ 1 -eq "$SKIP_START" ] ; then
		echo -e "\n\n ========= SKIPPING $NAME SERVICE STARTUP AS REQUESTED =========\n"
	else
		echo "Enabling and starting $NAME core service... (${DWARFG_NAM}.service)"
		if (run_app_svc) ; then 
			echo "Done - $NAME is deployed."
		else
			fail_fat "Unable to run $NAME service."
		fi
	fi
	event_log 307 MSG="$NAME version $VERSION was successfully deployed."
	[ "$GET_NONDEB" -eq 1 ] && {
		echo "Attempting to install non-Debian SW components..."
		echo "    (1) Grafana installation and integration: " && /usr/bin/dwarfg_grafint
		echo "    (2) PureSNMP library and SSHWifty/WebSSH: " && /usr/bin/dwarfg_inst_pyext
	}
	exit 0
fi
